/*
 * Copyright (c) Microsoft Corporation. All rights reserved. Licensed under the MIT license.
 * See LICENSE in the project root for license information.
 */

/* global global, Office, self, window */
const g = getGlobal();
var mailboxItem;

Office.onReady(() => {
	// If needed, Office.js is ready to be called
	mailboxItem = Office.context.mailbox.item;
	
	console.log("mailboxItem Loaded!");
});

function getGlobal() {
  return typeof self !== "undefined"
    ? self
    : typeof window !== "undefined"
    ? window
    : typeof global !== "undefined"
    ? global
    : undefined;
}

// The add-in command functions need to be available in global scope
g.listFilesProcessing = [];

function onMessageComposeHandler(event) {
	g.listFilesProcessing = [];
}

async function onMessageAttachmentsChangedHandler(event) {
	if(g.listFilesProcessing.length > 0 && g.listFilesProcessing.includes(event.attachmentDetails.name) && event.attachmentStatus == "added") {
		g.listFilesProcessing = g.listFilesProcessing.filter(e => e !== event.attachmentDetails.name);
		
	} else if(event.attachmentStatus == "added") {
		console.log("File added by Outlook: " + event.attachmentDetails.name);
		
		let fileContent = await getAttachmentContent(event.attachmentDetails);
	  
		if(fileContent.message === 'Success') {
			//We simulate the metadata processing of the file
			let processedFile = await processMetadata(fileContent.attachContent, fileContent.attachName);
						
			//Si el procesado de metadatos es correcto borro el fichero adjunto original
			if(processedFile.message === 'Success') {
				console.log("Successfully processed metadata: " + event.attachmentDetails.name );
			
				let removeResult = await removeAttachment(fileContent.attachName, fileContent.attachID);
				
				if(removeResult != null && removeResult.message != null && removeResult.message === 'Success') {					
					//We add the name of the file so as not to process it recursively
					g.listFilesProcessing.push(fileContent.attachName);
					
					await addBase64Attachment(processedFile.blobFile, fileContent.attachName);
					
				} else
					console.log(`Error Funcion removeAttachment: ${removeResult.message} to file: ${fileContent.attachName}`);
			}
		}
	}
	
	// Be sure to indicate when the add-in command function is complete
	event.completed();
}
	
function getAttachmentContent(attachment) {
		return new Promise(resolve => {
			mailboxItem.getAttachmentContentAsync(
				attachment.id,
				{ asyncContext : { nombreFichero: attachment.name, attachID: attachment.id} },
				function(result) {
					if (result.status !== Office.AsyncResultStatus.Succeeded) {
						console.log(`Error downloading the file: ${result.asyncContext.nombreFichero} :::: ${result.error.message}`);
						
						resolve({ message: result.error.message});
						
					} else if(result.value.format === Office.MailboxEnums.AttachmentContentFormat.Base64) {
						console.log(`Attachment content loaded successfully: ${result.asyncContext.nombreFichero}`);
						
						resolve({ message: 'Success', attachID: result.asyncContext.attachID, attachName: result.asyncContext.nombreFichero, attachContent: result.value.content });
						
					} else {
						console.log(`Error downloading the file: ${result.asyncContext.nombreFichero} :::: FileType :::: ${result.value.format}`);
						
						resolve({ message: `Error downloading the file: ${result.asyncContext.nombreFichero} :::: FileType :::: ${result.value.format}`});
					}
				}
			);
		});
	}
	
	function processMetadata(fileContent, attachmentName) {
		return new Promise(resolve => {
			bstr = atob(fileContent), 
            n = bstr.length, 
            u8arr = new Uint8Array(n);
            
			while(n--) {
				u8arr[n] = bstr.charCodeAt(n);
			}
					
			//We simulate the processing of the file...
			
			resolve({
				message: 'Success',
				blobFile: new File([u8arr], attachmentName),
				responseCode: 200,
				fileName: attachmentName
			});
		
		});
	}	
	
	function removeAttachment(attachmentName, attachmentID) {
		console.log('Start call delete file: ' + attachmentName);
		
		return new Promise(resolve => {
			try {
				mailboxItem.removeAttachmentAsync(
					attachmentID,
					{ asyncContext : { nombreFichero: attachmentName} },
					function(result) {
						if (result.status !== Office.AsyncResultStatus.Succeeded) {
							console.log(`Error deleting file: ${result.asyncContext.nombreFichero} :::: ${result.error.message}`);
									
							resolve({ message: result.error.message });
									
						} else {
							console.log(`Attachment removed successfully: ${result.asyncContext.nombreFichero}`);
									
							resolve({ message: 'Success' });
						}
					}
				);
			} catch (e) {
				console.log("Error deleting file ---> " + e.message);
				let mensajeError = (e.message != null && e.message.length > 0) ? e.message : '---';
				resolve({ message: mensajeError });
			}
		});
	}
	
	function addBase64Attachment(blob, attachmentName) {
		return new Promise(resolve => {            
			if(blob instanceof String) {
				try {
					mailboxItem.addFileAttachmentAsync(
						blob.valueOf(),
						attachmentName,
						{ isInline: false },
						function(result) {
							if (result.status === Office.AsyncResultStatus.Succeeded) {
								console.log(`Attachment added successfully: ${attachmentName}`);
									
								resolve({ message: 'Success'});
								  
							} else {
								console.log(`Attachment added Error: ${attachmentName} :::: ${result.error}`);
								resolve({ message: result.error.message });
							}
						});
				} catch (e) {
					//console.log("Error in UpdateProgessBar ---> " + e.message);
					let mensajeError = (e.message != null && e.message.length > 0) ? e.message : '---';
					resolve({ message: mensajeError });
				}
					
			} else {
				let reader = new FileReader();
				reader.readAsDataURL(blob);
				
				reader.onloadend = function() {
					try {
						mailboxItem.addFileAttachmentFromBase64Async(
							reader.result.split(',')[1],
							attachmentName,
							{ isInline: false },
							function(result) {
								if (result.status === Office.AsyncResultStatus.Succeeded) {
									console.log(`Attachment added successfully: ${attachmentName}`);
									
									resolve({ message: 'Success'});
								  
								} else {
									console.log(`Attachment added Error: ${attachmentName} :::: ${result.error}`);
									resolve({ message: result.error.message });
								}
						});
					} catch (e) {
						//console.log("Error in UpdateProgessBar ---> " + e.message);
						let mensajeError = (e.message != null && e.message.length > 0) ? e.message : '---';
						resolve({ message: mensajeError });
					}
				}
			}
		});
	}

// 1st parameter: FunctionName of LaunchEvent in the manifest; 2nd parameter: Its implementation in this .js file.
Office.actions.associate("onMessageComposeHandler", onMessageComposeHandler);
Office.actions.associate("onMessageAttachmentsChangedHandler", onMessageAttachmentsChangedHandler);